﻿using Hims.Api.Utilities;
using Hims.Domain.Helpers;
using Hims.Domain.Services;
using Hims.Shared.DataFilters;
using Hims.Shared.EntityModels;
using Hims.Shared.Library.Enums;
using Microsoft.AspNetCore.Mvc;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;

namespace Hims.Api.Controllers
{

    /// <inheritdoc />
    [Route("api/diet-condition")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class DietConditionsController : BaseController
    {

        /// <summary>
        /// The auditlog services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <summary>
        /// The auditlog services.
        /// </summary>
        private readonly IAESHelper aeshelper;


        private readonly IDietConditionHeaderService dietConditionHeaderService;

        public DietConditionsController(
           IAuditLogService auditLogServices,
           IDietConditionHeaderService dietConditionHeaderServices,
           IAESHelper aeshelper)
        {

            this.auditLogServices = auditLogServices;
            this.dietConditionHeaderService = dietConditionHeaderServices;
            this.aeshelper = aeshelper;
        }

        [HttpPost]
        [Route("fetch-items")]
        public async Task<ActionResult> FetchItems([FromBody] DietConditionsModel model)
        {
            var response = await this.dietConditionHeaderService.FetchItemsAsync(model);
            return this.Success(response);
        }
        [HttpPost]
        [Route("fetch-measure")]
        public async Task<ActionResult> FetchMeasure([FromBody] DietConditionsModel model)
        {
            var response = await this.dietConditionHeaderService.FetchMeasureAsync(model);
            return this.Success(response);

        }

        [HttpPost]
        [Route("add")]
        public async Task<ActionResult> InsertAsync([FromBody] ModifyConditionsRequest model)
        {
            model = (ModifyConditionsRequest)EmptyFilter.Handler(model);
            var response = 0;
            if (model.DietSlots.ConditionId == 0)
            {
                response = await this.dietConditionHeaderService.InsertItemsAsync(model.DietSlots, model.DietConditions);

            }
            else
            {
                response = await this.dietConditionHeaderService.UpdateItemsAsync(model.DietSlots, model.DietConditions);
            }
            var auditLogModel = new AuditLogModel
            {
                AccountId = model.DietSlots.CreatedBy,
                LogTypeId = (int)LogTypes.DietConditions,
                LogFrom = (short)model.DietSlots.LoginRoleId,
                LogDate = DateTime.UtcNow.AddMinutes(330),
                LogDescription = $@"<b>{model.DietSlots.CreatedByName}</b> has  {(model.DietSlots.ConditionId == 0 ? "Added" : "Updated")} <b>{model.DietSlots.ConditionName}</b>successfully",
                LocationId = Convert.ToInt32(model.DietSlots.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);
            return this.Success(response);

        }
        [HttpPut]
        [Route("update")]
        public async Task<ActionResult> UpdateAsync([FromBody] DietConditionsModel model)
        {
            model = (DietConditionsModel)EmptyFilter.Handler(model);

            var response = await this.dietConditionHeaderService.UpdateAsync(model);
            if (response > 0)
            {
                var auditLog = new AuditLogModel
                {
                    AccountId = model.CreatedBy,
                    LogTypeId = (int)LogTypes.DietConditions,
                    LogDate = DateTime.Now,
                    //LogFrom = short.Parse(model.LoginRoleId.ToString()),
                    LogDescription = $"{model.CreatedByName} has updated <b>Shift Name</b> of <strong>{model.ConditionName}</strong> successfully."

                };
                await this.auditLogServices.LogAsync(auditLog);
            }

            return this.Success(response);
        }

        [HttpPost]
        [Route("modify-status")]
        public async Task<ActionResult> ModifyStatusAsync([FromBody] DietConditionsModel model)
        {
            model = (DietConditionsModel)EmptyFilter.Handler(model);
            var response = await this.dietConditionHeaderService.ModifyStatusAsync(model);
            if (response > 0)
            {
                var auditLogModel = new AuditLogModel
                {
                    AccountId = model.ModifiedBy,
                    LogTypeId = (int)LogTypes.DietConditions,
                    LogDate = DateTime.Now,
                    // LogFrom = short.Parse(model.LoginRoleId.ToString()),
                    LogDescription = model.Active == true ? $@"{model.ModifiedByName} has Activated status of diet <br/>name: <b>{model.ConditionName}</b> on {DateTime.UtcNow.AddMinutes(330)}" : $@"{model.ModifiedByName} has Deactivated status of diet item <br/>name: '{model.ConditionName}' on {DateTime.UtcNow.AddMinutes(330)}",

                };
                await this.auditLogServices.LogAsync(auditLogModel);
            }
            return this.Success(response);

        }

        [HttpPost]
        [Route("fetch-diet-packages")]
        public async Task<ActionResult> FetchDietPackageAsync([FromBody] DietConditionsDetailModel model)
        {
            model = (DietConditionsDetailModel)EmptyFilter.Handler(model);
            var response = await this.dietConditionHeaderService.FetchDietPackageAsync(model);
            foreach (var record in response)
            {
                record.EncryptedConditionId = this.aeshelper.Encrypt(record.ConditionId.ToString());
            }
            return this.Success(response);

        }
        [HttpPost]
        [Route("modify-status-package")]
        public async Task<ActionResult> ModifyPackageStatusAsync([FromBody] DietConditionsModel model)
        {
            model = (DietConditionsModel)EmptyFilter.Handler(model);
            var response = await this.dietConditionHeaderService.ModifyPackageStatusAsync(model);
            var auditLogModel = new AuditLogModel
            {
                AccountId = model.ModifiedBy,
                LogTypeId = (int)LogTypes.DietConditions,
                LogFrom = (short)model.LoginRoleId,
                LogDate = DateTime.UtcNow.AddMinutes(330),
                LogDescription = $@"<b>{model.ModifiedByName}</b> has {((bool)model.Active ? "Activated" : "Deactivated")} the ConditionName <b> {model.ConditionName}</b>  successfully",
                LocationId = Convert.ToInt32(model.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);
            return this.Success(response);
        

        }
        [HttpGet]
        [Route("view-package")]
        public async Task<ActionResult> ViewAsync(string id)
        {
            var packageModuleId = Convert.ToInt32(this.aeshelper.Decrypt(id));
            var result = await this.dietConditionHeaderService.ViewAsync(packageModuleId);
            return this.Success(result);
        }
        public class ModifyConditionsRequest
        {
            public DietSlotsModel DietSlots { get; set; }

            public List<DietConditionsDetailModel> DietConditions { get; set; }

        }

    }
}
